local EffectEvents = require "effectevents"
local SGCommon = require "stategraphs.sg_common"
local playerutil = require "util.playerutil"
local monsterutil = require "util.monsterutil"

local PIERCE_JUMP_SPEED = 20

local function OnClawHitboxTriggered(inst, data)
	SGCommon.Events.OnHitboxTriggered(inst, data, {
		attackdata_id = "claw",
		hitstoplevel = HitStopLevel.LIGHT,
		hitflags = Attack.HitFlags.LOW_ATTACK,
		pushback = 1,
		hitstun_anim_frames = 5,
		combat_attack_fn = "DoBasicAttack",
		hit_fx = monsterutil.defaultAttackHitFX,
		hit_fx_offset_x = 0.5,
	})
end

local function OnBiteHitboxTriggered(inst, data)
	SGCommon.Events.OnHitboxTriggered(inst, data, {
		attackdata_id = "bite",
		hitstoplevel = HitStopLevel.LIGHT,
		hitflags = Attack.HitFlags.LOW_ATTACK,
		pushback = 1,
		hitstun_anim_frames = 5,
		combat_attack_fn = "DoKnockbackAttack",
		hit_fx = monsterutil.defaultAttackHitFX,
		hit_fx_offset_x = 0.5,
	})
end

local function OnPierceHitboxTriggered(inst, data)
	SGCommon.Events.OnHitboxTriggered(inst, data, {
		attackdata_id = "pierce",
		hitstoplevel = HitStopLevel.LIGHT,
		hitflags = Attack.HitFlags.LOW_ATTACK,
		pushback = 1,
		hitstun_anim_frames = 5,
		combat_attack_fn = "DoBasicAttack",
		hit_fx = monsterutil.defaultAttackHitFX,
		hit_fx_offset_x = 0.5,
	})

	SGCommon.Fns.SetMotorVelScaled(inst, PIERCE_JUMP_SPEED * 0.25)
end

local function ChooseIdleBehavior(inst)
	if not inst.components.timer:HasTimer("idlebehavior_cd") then
		local threat = playerutil.GetRandomLivingPlayer()
		if (not threat) then
			local rand_chance = math.random()
			if (rand_chance > 0.66) then
				inst.sg:GoToState("behavior1")
			elseif (rand_chance > 0.33) then
				inst.sg:GoToState("behavior2")
			end
			inst.components.timer:StartTimer("idlebehavior_cd", 10)
			return rand_chance > 0.33
		end
	end
	return false
end

local function OnDeath(inst, data)
	--Spawn death fx
	EffectEvents.MakeEventFXDeath(inst, data.attack, "death_mothball")

	inst.components.lootdropper:DropLoot()
end

local events =
{
}
monsterutil.AddMonsterCommonEvents(events,
{
	ondeath_fn = OnDeath,
})
monsterutil.AddOptionalMonsterEvents(events,
{
	idlebehavior_fn = ChooseIdleBehavior,
	spawn_battlefield = true,
})
SGCommon.Fns.AddCommonSwallowedEvents(events)

local states =
{
	State({
		name = "intro_bandicoot",
		tags = { "busy", "nointerrupt" },

		onenter = function(inst, attack_fn)
			inst.AnimState:PlayAnimation("intro_bandicoot")
			inst.Physics:StartPassingThroughObjects()
		end,

		timeline =
		{
			-- Code Generated by PivotTrack.jsfl
			FrameEvent(163, function(inst) inst.Physics:MoveRelFacing(86/150) end),
			FrameEvent(165, function(inst) inst.Physics:MoveRelFacing(162/150) end),
			FrameEvent(167, function(inst) inst.Physics:MoveRelFacing(134/150) end),
			FrameEvent(169, function(inst) inst.Physics:MoveRelFacing(103/150) end),
			FrameEvent(171, function(inst) inst.Physics:MoveRelFacing(87/150) end),
			FrameEvent(173, function(inst) inst.Physics:MoveRelFacing(70/150) end),
			FrameEvent(175, function(inst) inst.Physics:MoveRelFacing(81/150) end),
			FrameEvent(177, function(inst) inst.Physics:MoveRelFacing(89/150) end),
			FrameEvent(179, function(inst) inst.Physics:MoveRelFacing(84/150) end),
			FrameEvent(180, function(inst) inst.Physics:MoveRelFacing(42/150) end),
			FrameEvent(181, function(inst) inst.Physics:MoveRelFacing(31/150) end),
			FrameEvent(182, function(inst) inst.Physics:MoveRelFacing(19/150) end),
		},

		events =
		{
			EventHandler("animover", function(inst)
				inst.sg:GoToState("idle")
			end),
		},

		onexit = function(inst)
			inst.Physics:StopPassingThroughObjects()
		end,
	}),

	State({
		name = "intro_groak1",
		tags = { "busy", "nointerrupt" },

		onenter = function(inst)
			inst.AnimState:PlayAnimation("behavior1")
			inst.sg.mem.start_pos = inst:GetPosition()
			inst.sg.mem.spitout_knockback = 1
			inst.sg.mem.spitout_angle = 20
		end,

		events =
		{
			EventHandler("animover", function(inst)
				inst.sg:GoToState("intro_groak3")
			end),
		},
	}),

	State({
		name = "intro_groak2",
		tags = { "busy", "nointerrupt" },

		onenter = function(inst)
			inst.AnimState:PlayAnimation("behavior2")
			inst.sg.mem.start_pos = inst:GetPosition()
			inst.sg.mem.spitout_knockback = 0.5
			inst.sg.mem.spitout_angle = -20
		end,

		events =
		{
			EventHandler("animover", function(inst)
				inst.sg:GoToState("intro_groak3")
			end),
		},
	}),

	State({
		name = "intro_groak3",
		tags = { "busy", "nointerrupt" },

		onenter = function(inst)
			inst.AnimState:PlayAnimation("idle", true)
		end,

		onexit = function(inst)
			inst.sg.mem.spitout_knockback = nil
			inst.sg.mem.spitout_angle = nil
		end,
	}),

	State({
		name = "behavior1",
		tags = { "busy", "nointerrupt" },

		onenter = function(inst)
			inst.AnimState:PlayAnimation("behavior1")
		end,

		events =
		{
			EventHandler("animover", function(inst)
				inst.sg:GoToState("idle")
			end),
		},
	}),

	State({
		name = "behavior2",
		tags = { "busy", "nointerrupt" },

		onenter = function(inst)
			inst.AnimState:PlayAnimation("behavior2")
		end,

		events =
		{
			EventHandler("animover", function(inst)
				inst.sg:GoToState("idle")
			end),
		},
	}),

	State({
		name = "pierce",
		tags = { "attack", "busy" },

		onenter = function(inst, target)
			inst.AnimState:PlayAnimation("pierce")
			if (target) then
				SGCommon.Fns.FaceTargetClampedAngle(inst, target, 50)
			end
		end,

		onupdate = function(inst)
			if inst.sg.statemem.hitting then
				inst.components.hitbox:PushBeam(-0.3, 1.4, 0.8, HitPriority.MOB_DEFAULT)
			end
		end,

		timeline =
		{
			FrameEvent(6, function(inst)
				SGCommon.Fns.SetMotorVelScaled(inst, PIERCE_JUMP_SPEED)
				inst.components.hitbox:StartRepeatTargetDelay()
				inst.sg.statemem.hitting = true
				inst.sg:AddStateTag("airborne")
				inst.Physics:StartPassingThroughObjects()
			end),
			FrameEvent(12, function(inst)
				inst.sg.statemem.hitting = false
				SGCommon.Fns.SetMotorVelScaled(inst, PIERCE_JUMP_SPEED * 0.5)
			end),
			FrameEvent(14, function(inst)
				inst.sg:RemoveStateTag("airborne")
				SGCommon.Fns.SetMotorVelScaled(inst, PIERCE_JUMP_SPEED * 0.1)
				inst.Physics:StopPassingThroughObjects()
			end),
			FrameEvent(16, function(inst)
				if (inst:HasTag("elite")) then
					inst.sg:GoToState("bite_pre")
				else
					SGCommon.Fns.SetMotorVelScaled(inst, PIERCE_JUMP_SPEED * 0.2)
					inst.sg:AddStateTag("airborne")
				end
			end),
			FrameEvent(20, function(inst)
				inst.Physics:Stop()
				inst.sg:RemoveStateTag("airborne")
			end),
		},

		events =
		{
			EventHandler("hitboxtriggered", OnPierceHitboxTriggered),
			EventHandler("animover", function(inst)
				inst.sg:GoToState("idle")
			end),
		},

		onexit = function(inst)
			inst.Physics:Stop()
			inst.components.attacktracker:CompleteActiveAttack()
			inst.components.hitbox:StopRepeatTargetDelay()
			inst.Physics:StopPassingThroughObjects()
		end,
	}),

	State({
		name = "bite_pre",
		tags = { "busy", "nointerrupt" },

		onenter = function(inst)
			inst.AnimState:PlayAnimation("elite_bite_pre")
		end,

		events =
		{
			EventHandler("animover", function(inst)
				inst.sg:GoToState("bite")
			end),
		},
	}),

	State({
		name = "bite",
		tags = { "busy", "nointerrupt" },

		onenter = function(inst)
			inst.AnimState:PlayAnimation("elite_bite")
			inst.sg.statemem.hitting = false
			inst.components.hitbox:StartRepeatTargetDelay()
			local target = inst.components.combat:GetTarget()
			if (target) then
				SGCommon.Fns.FaceTarget(inst, target, true)
			end
		end,

		timeline =
		{
			FrameEvent(2, function(inst)
				inst.Physics:StartPassingThroughObjects()
				SGCommon.Fns.SetMotorVelScaled(inst, 12)
				inst.sg.statemem.hitting = true
			end),
			FrameEvent(6, function(inst)
				inst.Physics:StopPassingThroughObjects()
				SGCommon.Fns.SetMotorVelScaled(inst, 4)
			end),
			FrameEvent(7, function(inst)
				inst.sg.statemem.hitting = false
			end),
			FrameEvent(8, function(inst)
				inst.Physics:Stop()
			end)
		},

		onupdate = function(inst)
			if (inst.sg.statemem.hitting) then
				inst.components.hitbox:PushBeam(0, 1.3, 1, HitPriority.MOB_DEFAULT)
			end
		end,

		events =
		{
			EventHandler("hitboxtriggered", OnBiteHitboxTriggered),
			EventHandler("animover", function(inst)
				inst.sg:GoToState("idle")
			end),
		},

		onexit = function(inst)
			inst.Physics:Stop()
			inst.components.hitbox:StopRepeatTargetDelay()
		end,
	}),

	State({
		name = "claw",
		tags = { "attack", "busy" },

		onenter = function(inst, target)
			inst.AnimState:PlayAnimation("claw")
			inst.sg.statemem.target = target
		end,

		timeline =
		{
			FrameEvent(12, function(inst)
				inst.components.hitbox:StartRepeatTargetDelay()
				inst.components.hitbox:PushBeam(0, 2, 1.25, HitPriority.MOB_DEFAULT)
			end),
			FrameEvent(13, function(inst)
				inst.components.hitbox:PushBeam(0, 2, 1.25, HitPriority.MOB_DEFAULT)
			end),
		},

		events =
		{
			EventHandler("hitboxtriggered", OnClawHitboxTriggered),
			EventHandler("animover", function(inst)
				inst.sg:GoToState("idle")
			end),
		},

		onexit = function(inst)
			inst.components.attacktracker:CompleteActiveAttack()
			inst.components.hitbox:StopRepeatTargetDelay()
		end,
	}),
}

SGCommon.States.AddAttackPre(states, "pierce")
SGCommon.States.AddAttackHold(states, "pierce")

SGCommon.States.AddAttackPre(states, "claw")
SGCommon.States.AddAttackHold(states, "claw")

SGCommon.States.AddSpawnBattlefieldStates(states,
{
	anim = "spawn",
	fadeduration = 0.33,
	fadedelay = 0,
	onenter_fn = function(inst)
		local vel = math.random(5, 8)
		SGCommon.Fns.SetMotorVelScaled(inst, vel)
		inst.Physics:StartPassingThroughObjects()
	end,
	timeline =
	{
		FrameEvent(0, function(inst) inst:PushEvent("leave_spawner") end),

		FrameEvent(18, function(inst)
			inst.Physics:Stop()
			inst.Physics:StopPassingThroughObjects()
		end),
	},
	onexit_fn = function(inst)
		inst.Physics:Stop()
		inst.Physics:StopPassingThroughObjects()
	end,
})

SGCommon.States.AddHitStates(states)
SGCommon.States.AddKnockbackStates(states,
{
	movement_frames = 7,
})
SGCommon.States.AddKnockdownStates(states,
{
	movement_frames = 12,
})
SGCommon.States.AddKnockdownHitStates(states)

SGCommon.States.AddIdleStates(states)
SGCommon.States.AddWalkStates(states)
SGCommon.States.AddTurnStates(states)

SGCommon.States.AddMonsterDeathStates(states)

SGRegistry:AddData("sg_mothball", states)

return StateGraph("sg_mothball", states, events, "idle")
